"use client";
import React, { useState } from "react";
import Link from "next/link";
import {
  ArrowUp,
  Facebook,
  Instagram,
  X,
  Youtube,
  MapPin,
  Phone,
  Mail,
  Clock,
  FeatherIcon,
} from "lucide-react";
import { useSettings } from "@/context/SettingsContext";
import WhatsappChatModal from "../ChatModel/WhatsappChatModal";
import Image from "next/image";
import { FaXTwitter } from "react-icons/fa6";
import { useCategories, useBranches } from "@/hooks/useData";

const iconMap = {
  instagram: <Instagram className="w-5 h-5" />,
  facebook: <Facebook className="w-5 h-5" />,
  twitter: <FaXTwitter className="w-5 h-5" />,
  x: <X className="w-5 h-5" />,
  youtube: <Youtube className="w-5 h-5" />,
};

export default function Footer() {
  const { settings } = useSettings();
  const [isWhatsappOpen, setIsWhatsappOpen] = useState(false);
  const { data, isLoading } = useCategories();
  const { data: branches, isLoading: branchesLoading } = useBranches();

  return (
    <footer className="relative bg-gradient-to-b from-brand-dark via-[#094044] to-[#051f21] text-white mt-20">
      {/* Decorative top border with golden accent */}
      <div className="absolute top-0 left-0 right-0 h-1 bg-gradient-to-r from-transparent via-brand-gold to-transparent"></div>

      {/* Subtle pattern overlay */}
      <div
        className="absolute inset-0 opacity-5"
        style={{
          backgroundImage: `url("data:image/svg+xml,%3Csvg width='60' height='60' viewBox='0 0 60 60' xmlns='http://www.w3.org/2000/svg'%3E%3Cg fill='none' fill-rule='evenodd'%3E%3Cg fill='%23ddbc5f' fill-opacity='1'%3E%3Cpath d='M36 34v-4h-2v4h-4v2h4v4h2v-4h4v-2h-4zm0-30V0h-2v4h-4v2h4v4h2V6h4V4h-4zM6 34v-4H4v4H0v2h4v4h2v-4h4v-2H6zM6 4V0H4v4H0v2h4v4h2V6h4V4H6z'/%3E%3C/g%3E%3C/g%3E%3C/svg%3E")`,
        }}
      ></div>

      <div className="relative max-w-[1600px] mx-auto px-6 py-16">
        {/* Main Footer Content */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-12 mb-16">
          {/* Brand Section */}
          <div className="lg:col-span-1 flex flex-col items-center lg:items-start">
            <div className="mb-6 relative">
              <div className="absolute inset-0 bg-brand-gold/10 blur-2xl rounded-full"></div>
              <img
                src={settings?.logo?.replace("/api", "")}
                alt="Selvakumar Jewellary"
                className="relative w-full max-w-[200px] mb-4 drop-shadow-2xl"
              />
            </div>
            <p className="text-gray-300 text-sm leading-relaxed mb-6 font-light text-center lg:text-left">
              Crafting timeless elegance with exquisite jewelry that celebrates
              life's precious moments.
            </p>
          </div>

          {/* Quick Links */}
          <div>
            <h3 className="text-lg font-semibold mb-6 font-cormorant uppercase tracking-wider relative inline-block">
              <span className="relative z-10 bg-gradient-to-r from-brand-gold to-[#f0d78c] bg-clip-text text-transparent">
                Quick Links
              </span>
              <div className="absolute bottom-0 left-0 w-12 h-0.5 bg-gradient-to-r from-brand-gold to-transparent"></div>
            </h3>

            <ul className="space-y-3">
              <li>
                <Link
                  href="/"
                  className="text-gray-300 hover:text-brand-gold transition-colors duration-300 flex items-center group text-sm"
                >
                  <span className="w-0 group-hover:w-2 h-px bg-brand-gold transition-all duration-300 mr-0 group-hover:mr-2"></span>
                  Home
                </Link>
              </li>

              {isLoading
                ? [0, 1].map((index) => (
                    <li key={index}>
                      <div className="h-3 w-24 rounded animate-pulse bg-gray-700"></div>
                    </li>
                  ))
                : data &&
                  data
                    ?.filter(
                      (cat) =>
                        cat.parent_id === null && cat.category_status === 1,
                    )
                    .map((cat) => (
                      <li key={cat.id}>
                        <Link
                          href={`/${cat.category_url}`}
                          className="text-gray-300 hover:text-brand-gold transition-colors duration-300 flex items-center group text-sm"
                        >
                          <span className="w-0 group-hover:w-2 h-px bg-brand-gold transition-all duration-300 mr-0 group-hover:mr-2"></span>
                          {cat.name}
                        </Link>
                      </li>
                    ))}

              <li>
                <Link
                  href="/about"
                  className="text-gray-300 hover:text-brand-gold transition-colors duration-300 flex items-center group text-sm"
                >
                  <span className="w-0 group-hover:w-2 h-px bg-brand-gold transition-all duration-300 mr-0 group-hover:mr-2"></span>
                  About
                </Link>
              </li>
              <li>
                <Link
                  href="/contact"
                  className="text-gray-300 hover:text-brand-gold transition-colors duration-300 flex items-center group text-sm"
                >
                  <span className="w-0 group-hover:w-2 h-px bg-brand-gold transition-all duration-300 mr-0 group-hover:mr-2"></span>
                  Contact
                </Link>
              </li>
            </ul>
          </div>

          {/* Contact Information */}
          <div>
            <h3 className="text-lg font-semibold mb-6 font-cormorant uppercase tracking-wider relative inline-block">
              <span className="relative z-10 bg-gradient-to-r from-brand-gold to-[#f0d78c] bg-clip-text text-transparent">
                Get In Touch
              </span>
              <div className="absolute bottom-0 left-0 w-12 h-0.5 bg-gradient-to-r from-brand-gold to-transparent"></div>
            </h3>

            <ul className="space-y-4">
              <li className="flex items-start gap-3 group">
                <div className="mt-1 p-2 bg-brand-gold/10 rounded-lg group-hover:bg-brand-gold/20 transition-colors">
                  <MapPin className="w-4 h-4 text-brand-gold" />
                </div>
                <div className="flex-1">
                  <p className="text-gray-300 text-sm leading-relaxed">
                    {settings?.address}
                  </p>
                </div>
              </li>

              <li className="flex items-start gap-3 group">
                <div className="mt-0.5 p-2 bg-brand-gold/10 rounded-lg group-hover:bg-brand-gold/20 transition-colors">
                  <Phone className="w-4 h-4 text-brand-gold" />
                </div>
                <div className="flex-1">
                  <Link
                    href={`tel:${settings.mobile}`}
                    className="text-gray-300 hover:text-brand-gold transition-colors text-sm block"
                  >
                    +91 {settings.mobile}
                  </Link>
                </div>
              </li>

              <li className="flex items-start gap-3 group">
                <div className="mt-0.5 p-2 bg-brand-gold/10 rounded-lg group-hover:bg-brand-gold/20 transition-colors">
                  <Mail className="w-4 h-4 text-brand-gold" />
                </div>
                <div className="flex-1">
                  <Link
                    href={`mailto:${settings.email}`}
                    className="text-gray-300 hover:text-brand-gold transition-colors text-sm block break-all"
                  >
                    {settings.email}
                  </Link>
                </div>
              </li>
            </ul>
          </div>

          {/* Social & Newsletter */}
          <div>
            <h3 className="text-lg font-semibold mb-6 font-cormorant uppercase tracking-wider relative inline-block">
              <span className="relative z-10 bg-gradient-to-r from-brand-gold to-[#f0d78c] bg-clip-text text-transparent">
                Stay Connected
              </span>
              <div className="absolute bottom-0 left-0 w-12 h-0.5 bg-gradient-to-r from-brand-gold to-transparent"></div>
            </h3>

            <p className="text-gray-300 text-sm mb-6 leading-relaxed">
              Follow us on social media for exclusive updates and offers.
            </p>

            {/* Social Links */}
            <div className="flex gap-3 mb-8">
              {settings?.social_links?.map((link, index) => {
                const icon = iconMap[link.platform.toLowerCase()] || (
                  <span className="text-xs">{link.platform}</span>
                );

                return (
                  <Link
                    key={index}
                    href={link.url}
                    target="_blank"
                    rel="noopener noreferrer"
                    className="group relative p-3 bg-gradient-to-br from-[#094044] to-[#051f21] border border-brand-dark hover:border-brand-gold transition-all duration-300 rounded-lg hover:shadow-lg hover:shadow-brand-gold/20"
                  >
                    <div className="absolute inset-0 bg-gradient-to-br from-brand-gold/0 to-brand-gold/0 group-hover:from-brand-gold/10 group-hover:to-brand-gold/20 rounded-lg transition-all duration-300"></div>
                    <div className="relative text-gray-400 group-hover:text-brand-gold transition-colors">
                      {icon}
                    </div>
                  </Link>
                );
              })}
            </div>

            {/* Newsletter (Optional)
            <div className="mt-6">
              <p className="text-xs text-gray-400 mb-3">
                Subscribe to our newsletter
              </p>
              <div className="flex gap-2">
                <input
                  type="email"
                  placeholder="Your email"
                  className="flex-1 px-4 py-2 bg-[#094044]/50 border border-brand-dark focus:border-brand-gold rounded-lg text-sm text-gray-300 placeholder-gray-500 focus:outline-none transition-colors"
                />
                <button className="px-4 py-2 bg-gradient-to-r from-brand-gold to-[#f0d78c] hover:from-[#f0d78c] hover:to-brand-gold text-brand-dark font-medium rounded-lg text-sm transition-all duration-300 hover:shadow-lg hover:shadow-brand-gold/30">
                  Subscribe
                </button>
              </div>
            </div> */}
          </div>
        </div>

        {/* Branches Section */}
        {branches && branches.rows && branches.rows.length > 0 && (
          <div className="mb-16">
            <h3 className="text-2xl font-semibold mb-8 font-cormorant uppercase tracking-wider text-center">
              <span className="relative z-10 bg-gradient-to-r from-brand-gold to-[#f0d78c] bg-clip-text text-transparent">
                Our Branches
              </span>
            </h3>
            
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
              {branchesLoading
                ? [0, 1, 2, 3].map((index) => (
                    <div
                      key={index}
                      className="h-40 rounded-lg animate-pulse bg-gray-700/50"
                    ></div>
                  ))
                : branches.rows.map((branch) => (
                    <div
                      key={branch.id}
                      className="group relative border border-brand-gold transition-all duration-300 rounded-lg p-5 hover:shadow-lg hover:shadow-brand-gold/20"
                    >
                      {/* Location Type Badge */}
                      <div className="mb-3">
                        <span
                          className={`inline-block px-3 py-1 rounded-full text-xs font-medium ${
                            branch.location_type === "branch"
                              ? "bg-brand-gold/20 text-brand-gold"
                              : branch.location_type === "franchise"
                              ? "bg-blue-500/20 text-blue-400"
                              : "bg-purple-500/20 text-purple-400"
                          }`}
                        >
                          {branch.location_type.charAt(0).toUpperCase() +
                            branch.location_type.slice(1)}
                        </span>
                      </div>

                      {/* Branch Name */}
                      {branch.name && (
                        <h4 className="text-brand-gold font-semibold text-base mb-3 font-cormorant">
                          {branch.name}
                        </h4>
                      )}

                      {/* Address */}
                      <div className="flex items-start gap-2 mb-3">
                        <MapPin className="w-4 h-4 text-brand-gold mt-0.5 flex-shrink-0" />
                        <p className="text-gray-300 text-xs leading-relaxed">
                          {branch.address}
                          {branch.city && `, ${branch.city}`}
                          {branch.state && `, ${branch.state}`}
                          {branch.pincode && ` - ${branch.pincode}`}
                        </p>
                      </div>

                      {/* Phone */}
                      {branch.phone && (
                        <div className="flex items-center gap-2">
                          <Phone className="w-4 h-4 text-brand-gold flex-shrink-0" />
                          <Link
                            href={`tel:${branch.phone}`}
                            className="text-gray-300 hover:text-brand-gold transition-colors text-xs"
                          >
                            {branch.phone}
                          </Link>
                        </div>
                      )}
                    </div>
                  ))}
            </div>
          </div>
        )}

        {/* Divider */}
        <div className="h-px bg-gradient-to-r from-transparent via-brand-gold/30 to-transparent mb-8"></div>

        {/* Bottom Bar */}
        <div className="flex flex-col md:flex-row justify-between items-center gap-4 text-xs text-gray-400">
          <p className="text-center md:text-left">
            &copy; {new Date().getFullYear()} Selvakumar Jewellery. All rights
            reserved.
          </p>

          <div className="flex items-center gap-4">
            {/* <Link
              href="/privacy-policy"
              className="hover:text-brand-gold transition-colors"
            >
              Privacy Policy
            </Link>
            <span className="text-brand-dark/50">|</span>
            <Link
              href="/terms"
              className="hover:text-brand-gold transition-colors"
            >
              Terms & Conditions
            </Link> */}
            {/* <span className="text-brand-dark/50">|</span> */}
            <Link
              className="hover:text-brand-gold transition-colors flex items-center gap-1"
              target="_blank"
              href="https://infodazz.org"
            >
              <FeatherIcon size={16} />
              Designed by{" "}
              <span className="font-medium text-brand-gold">Infodazz</span>
            </Link>
          </div>
        </div>
      </div>

      {/* Floating Action Buttons */}
      <div className="" onClick={() => setIsWhatsappOpen(true)}>
        <div className="w-14 h-14 fixed right-5 bottom-24 cursor-pointer z-50 hover:scale-110 transition-transform duration-300 drop-shadow-2xl">
          <Image
            src="/images/whatsapp.png"
            alt="whatsapp"
            fill
            sizes="100%"
            className="animate-pulse"
          />
        </div>
      </div>

      <button
        className="fixed cursor-pointer right-5 bottom-5 p-3 bg-gradient-to-br from-brand-gold to-[#f0d78c] hover:from-[#f0d78c] hover:to-brand-gold text-brand-dark rounded-full shadow-lg hover:shadow-brand-gold/50 transition-all duration-300 z-50 group hover:scale-110"
        onClick={() => window.scrollTo({ top: 0, behavior: "smooth" })}
        aria-label="Scroll to top"
      >
        <ArrowUp
          size={20}
          className="group-hover:-translate-y-1 transition-transform"
        />
      </button>

      <div className="fixed right-6 bottom-16 -translate-y-1/2 z-[100]">
        <WhatsappChatModal
          isOpen={isWhatsappOpen}
          onClose={() => setIsWhatsappOpen(false)}
        />
      </div>
    </footer>
  );
}
