"use client";

import React, { useEffect, useState } from "react";
import { Controller, useForm } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";
import * as yup from "yup";
import { useMutation, useQuery, useQueryClient } from "@tanstack/react-query";

import CustomInput from "@/components/ui/FormInput/CustomInput";

import { CustomDropZone, CustomToggle } from "@/components/ui/FormInput/Inputs";

import { Button } from "@/components/ui/button";
import MultiSelect from "@/components/ui/FormInput/MultiSelectNew";
import DividerWithText from "@/components/ui/FormInput/DividerWithText";
import { addCta, editCta } from "@/Services/Cta/ApiCta";
import { getProductCollections } from "@/Services/ProductCollections/ApiProductCollections";
import { getSiteMap } from "@/Services/SiteMap/ApiSiteMap";
import { Loader2 } from "lucide-react";
import { getCategories } from "@/Services/Categories/ApiCategory";
import { useModal } from "@/context/ModalProvider";

const schema = yup.object().shape({
  // image: yup.mixed().required("Banner Image is required"),
  image: yup.object({
    image: yup
      .array()
      .of(yup.mixed().required("Image is required"))
      .min(1, "Image is required")
      .required("Image is required"),
  }),
  // title: yup.string().required("Banner Title is required"),
  title: yup.string().nullable(),
  description: yup.string().nullable(),
  status: yup.string().default(true),
  show_button: yup.boolean().default(false),
  show_content: yup.boolean().default(false),
  show_description: yup.boolean().default(false),
  is_parallax: yup.boolean().default(false),
  button_link: yup.string().nullable(),
  alignment: yup.number().default(0),
});

if (!String.prototype.toCapitalized) {
  String.prototype.toCapitalized = function () {
    return this.replace(/[-_]/g, " ")
      .trim()
      .split(" ")
      .map((word) => word.charAt(0).toUpperCase() + word.slice(1).toLowerCase())
      .join(" ");
  };
}

const AddCta = ({ type = "add", editData, onClose }) => {
  const { closeModal } = useModal();
  const queryClient = useQueryClient();
  const {
    control,
    handleSubmit,
    reset,
    setValue,
    watch,
    formState: { errors },
  } = useForm({
    resolver: yupResolver(schema),
    defaultValues: {
      show_button: false,
      show_content: true,
      show_description: false,
      is_parallax: false,
      status: 1,
      alignment: 0,
    },
  });
  const [imageFile, setImageFile] = useState({
    image: editData?.image ? [editData.image] : [],
  });

  useEffect(() => {
    setValue("image", imageFile);
  }, [imageFile, setValue]);

  useEffect(() => {
    if (type === "edit" && editData) {
      Object.keys(editData).forEach((key) => {
        if (key === "visibility" && typeof editData.visibility === "object") {
          const visibility = editData?.visibility;
          Object?.keys(visibility)?.forEach((vKey) => {
            setValue(vKey, visibility[vKey] ? true : false);
          });
        } else {
          setValue(key, editData[key]);
        }
      });

      if (editData.image) {
        setImageFile({ image: [editData.image] });
      }
    }
  }, [type, editData, setValue]);

  const { data: sitemap = [], isLoading: isSiteMapLoading } = useQuery({
    queryKey: ["sitemap"],
    queryFn: () => getSiteMap(),
  });
  const { data: collections = [], isLoading: isCollectionLoading } = useQuery({
    queryKey: ["products_collections"],
    queryFn: () => getProductCollections(),
  });
  const { data: categories = [], isLoading: isCategoryLoading } = useQuery({
    queryKey: ["categories"],
    queryFn: () => getCategories(0, 10, false, true),
  });

  const addMutation = useMutation({
    mutationFn: addCta,
    onSuccess: () => {
      queryClient.invalidateQueries("cta");
      reset();
      setImageFile(false);
    },
  });

  const editMutation = useMutation({
    mutationFn: (data) => editCta(data, editData.id),
    onSuccess: () => {
      queryClient.invalidateQueries("cta");
      reset();
      setImageFile(false);
      onClose();
    },
  });

  const submitHandler = (data) => {
    const formData = new FormData();
    formData.append("folder", "cta");
    Object.entries(data).forEach(([key, value]) => {
      if (key === "image") {
        if (value?.image?.[0]) {
          formData.append("image", value.image[0]);
        } else {
          formData.append("image", JSON.stringify({ image: [] }));
        }
      } else {
        formData.append(key, value);
      }
    });

    type === "add"
      ? addMutation.mutate(formData)
      : editMutation.mutate(formData);
  };

  if (isSiteMapLoading || isCollectionLoading || isCategoryLoading) {
    return (
      <div className="flex items-center justify-center h-48">
        <Loader2 className="h-8 w-8 animate-spin" />
      </div>
    );
  }

  return (
    <div className="bg-white pt-0 w-full border border-gray-200 rounded-md ">
      <div className="flex justify-between items-center border-b p-4 bg-brand-dark/5">
        <h2 className="text-xl font-semibold text-brand-dark">
          {type === "add" ? "Add" : "Edit"} CTA
        </h2>
      </div>

      <form onSubmit={handleSubmit(submitHandler)} className="p-6 space-y-6">
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-2 gap-6">
          {/* <CustomDropZone
            name="image"
            label="CTA Image"
            number_of_images={1}
            type={type}
            errors={errors.image}
            fileType="image"
            image_size={2 * 1024 * 1024}
            uploadedFiles={imageFile}
            setUploadedFiles={setImageFile}
            showIcon={false}
            required
          /> */}

          <CustomDropZone
            name="image"
            label="CTA Image"
            number_of_images={1}
            errors={errors.image}
            fileType={"image"}
            image_size={1024 * 1024 * 2}
            uploadedFiles={imageFile}
            setUploadedFiles={setImageFile}
            showIcon={false}
            required
          />
          <DividerWithText
            text="Content Details"
            underline
            className="col-span-full"
            showLines
            textClassName="text-brand-dark"
            position="start"
            lineColor="border-brand-dark"
            component={
              <CustomToggle
                options={[
                  { name: "show_content", label: "Show Content" },
                  { name: "is_parallax", label: "Enable Parallax" },
                ]}
                control={control}
                errors={errors}
                showTooltip
                showLabel={true}
              />
            }
          />
          <Controller
            name="title"
            control={control}
            render={({ field }) => (
              <CustomInput
                type="textarea"
                maxLength={25}
                label="Cta Title"
                placeholder="Enter your Title"
                {...field}
                errors={errors}
                helperText="Maximum 25 characters"
              />
            )}
          />

          <Controller
            name="description"
            control={control}
            render={({ field }) => (
              <CustomInput
                type="textarea"
                maxLength={70}
                label="Description"
                placeholder="Enter your description"
                {...field}
                errors={errors}
                helperText="Maximum 70 characters"
                labelComponent={
                  <CustomToggle
                    options={[
                      { name: "show_description", label: "Show Description" },
                    ]}
                    control={control}
                    errors={errors}
                    showTooltip
                    showLabel={false}
                  />
                }
              />
            )}
          />

          <Controller
            name="alignment"
            control={control}
            render={({ field }) => (
              <MultiSelect
                {...field}
                label="Content Alignment"
                options={[
                  { value: 0, label: "Center" },
                  { value: 1, label: "Left" },
                  { value: 2, label: "Right" },
                ]}
                errors={errors}
              />
            )}
          />

          {/* <Controller
            name="button_link"
            control={control}
            render={({ field }) => {
              const siteMapOptions = (sitemap?.rows || []).map((route) => ({
                label: route.url.toCapitalized(),
                value: route.url,
              }));

              const collectionOptions = (collections?.rows || []).map(
                (col) => ({
                  label: col.name.toCapitalized(),
                  value: col.url,
                })
              );

              const mergedOptions = [...siteMapOptions, ...collectionOptions];

              return (
                <MultiSelect
                  {...field}
                  label="Button Link"
                  options={mergedOptions}
                  errors={errors}
                  isSearchable
                  placeholder="Select a page or collection"
                  labelComponent={
                    <CustomToggle
                      options={[{ name: "show_button", label: "Show Button" }]}
                      control={control}
                      errors={errors}
                      showTooltip
                      showLabel={false}
                    />
                  }
                />
              );
            }}
          /> */}

          <Controller
            name="button_link"
            control={control}
            render={({ field }) => {
              const siteMapOptions = (sitemap?.rows || []).map((route) => ({
                // label: route.url?.toCapitalized?.() || route.url,
                label: route.label,
                value: route.url,
              }));

              const collectionOptions = (collections?.rows || []).map(
                (col) => ({
                  label: col.name?.toCapitalized?.() || col.name,
                  value: col.url,
                })
              );

              const categoryOptions = (categories?.rows || []).map((cat) => {
                const label =
                  cat.category_url
                    ?.split("/")
                    .map((part) => part.toCapitalized?.() || part)
                    .join(" → ") || cat.name;

                return {
                  label,
                  value: cat.category_url,
                };
              });

              const mergedOptions = [
                ...siteMapOptions,
                ...collectionOptions,
                ...categoryOptions,
              ];

              return (
                <MultiSelect
                  {...field}
                  label="Button Link"
                  options={mergedOptions}
                  errors={errors}
                  isSearchable
                  placeholder="Select a page, collection or category"
                  labelComponent={
                    <CustomToggle
                      options={[{ name: "show_button", label: "Show Button" }]}
                      control={control}
                      errors={errors}
                      showTooltip
                      showLabel={false}
                    />
                  }
                />
              );
            }}
          />

          {/* <Controller
            name="button_link"
            control={control}
            render={({ field }) => (
              <CustomInput
                type="text"
                label="Button Link"
                placeholder="Button Link"
                {...field}
                errors={errors}
                labelComponent={
                  <CustomToggle
                    options={[{ name: "show_button", label: "Show Button" }]}
                    control={control}
                    errors={errors}
                    showTooltip
                    showLabel={false}
                  />
                }
              />
            )}
          /> */}
        </div>

        {/* <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <Controller
            name="title"
            control={control}
            render={({ field }) => (
              <CustomInput
                type="textarea"
                label="Banner Title"
                placeholder="Enter your Title"
                {...field}
                errors={errors}
                required
              />
            )}
          />

          <Controller
            name="description"
            control={control}
            render={({ field }) => (
              <CustomInput
                type="textarea"
                label="Description"
                placeholder="Enter your description"
                {...field}
                errors={errors}
              />
            )}
          />
        </div> */}

        <div className="flex justify-end">
          <Button
            onClick={() => {
              reset();
              setBannerImage(false);
            }}
            type="reset"
            className="mr-2 bg-red-700"
          >
            Reset
          </Button>
          <Button type="submit" variant="dark">
            Save
          </Button>
        </div>
      </form>
    </div>
  );
};

export default AddCta;
